package easik.sketch.util.graph;

import java.awt.Color;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GradientPaint;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.GridLayout;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.Iterator;
import java.util.Map;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.border.Border;

import org.jgraph.JGraph;
import org.jgraph.graph.CellView;
import org.jgraph.graph.CellViewRenderer;
import org.jgraph.graph.DefaultGraphCell;
import org.jgraph.graph.EdgeView;
import org.jgraph.graph.GraphConstants;
import org.jgraph.graph.VertexView;

import easik.Easik;
import easik.sketch.attribute.EntityAttribute;
import easik.sketch.attribute.UniqueKey;
import easik.sketch.vertex.EntityNode;

/**
 * A view class for the entity cell type.
 * 
 * @author William Belanger 2006
 * @author Kevin Green 2006
 * @since 2006-07-25 Kevin Green
 * @version 2006-07-26 Kevin Green
 */
public class EntityView extends VertexView {
	/** The renderer for this view */
	public static transient EntityRenderer _renderer = new EntityRenderer();

	/**
	 * Creates a new entity view.
	 */
	public EntityView() {
		super();
	}

	/**
	 * Sets the cell that this view will be used to render.
	 * 
	 * @param cell The cell to be rendered by this view.
	 */
	public EntityView(Object cell) {
		super(cell);
	}
	
	/**
	 * Returns the current renderer
	 * @return The current renderer
	 */
	public CellViewRenderer getRenderer() {
		return _renderer;
	}
	
	/**
	 * Overridden method to return the proper perimeter point for edges to use
	 * 
	 * @param edge The edge
	 * @param source The source of the edge
	 * @param p The desitination of the edge
	 * @return The perimeter intersection point
	 */
	public Point2D getPerimeterPoint(EdgeView edge, Point2D source, Point2D p) {
		return ((EntityRenderer) getRenderer()).getPerimeterPoint(this, source, p);
	}
	
    /**
	 * The renderer for the circle vertices.
	 */
	public static class EntityRenderer extends JPanel implements CellViewRenderer {

		/**
		 * The graph
		 */
		transient protected JGraph graph = null;
		/**
		 * The gradient color of the entity
		 */
    	transient protected Color gradientColor = null;
    	/**
    	 * If the entity cell has focus or not
    	 */
    	transient protected boolean hasFocus;
    	/**
    	 * If the entity cell is selected
    	 */
    	transient protected boolean selected;
    	/**
    	 * If the graph is in preview mode
    	 */
    	transient protected boolean preview;
    	/**
    	 * The label of the entity
    	 */
    	transient protected JLabel _entityLabel;
    	/**
    	 * The entity pane (Used for main label)
    	 */
    	transient protected JPanel _entity;
    	/**
    	 * The attributes pane
    	 */
    	transient protected JPanel _attributes;
    	/**
    	 * The unique keys pane
    	 */
    	transient protected JPanel _uniqueKeys;
    	
		/**
		 * Creates a new circle renderer.
		 */
		public EntityRenderer() {
			super();
		}
		
		/**
		 * Returns the renderer component after initializing it
		 * @param graph The graph
		 * @param view The cell view
		 * @param sel If the view is selected or not
		 * @param focus If the view has focus or not
		 * @param preview If the graph is in preview mode or not
		 * @return The renderer component fully initialized
		 */
        public Component getRendererComponent(JGraph graph, CellView view, boolean sel, boolean focus, boolean preview) {
            this.graph = graph;
            this.selected = sel;
            this.preview = preview;
            this.hasFocus = focus;
            
            Map attributes = view.getAllAttributes();
            installAttributes(graph, attributes);
            
            //Initialize panel
            this.removeAll();
            this.setLayout(new BoxLayout(this, BoxLayout.PAGE_AXIS));
            EntityNode myNode = (EntityNode) Easik.getInstance().getFrame().getSketch().getAdapter().getVertexFromCell((DefaultGraphCell)view.getCell());
            
            //Add in entity label
            this.add(_entity = new JPanel(new GridLayout(1,1)));
            _entity.add(_entityLabel = new JLabel(myNode.getName(), JLabel.CENTER));
            _entity.setForeground(Easik.getInstance().getIni().getENTITY_FG_COLOR());
            _entity.setBackground(Easik.getInstance().getIni().getENTITY_BG_COLOR());
            _entityLabel.setForeground(this.getForeground());
            
            //Add in attributes
            this.add(_attributes = new JPanel(new GridLayout(0,1)));
            _attributes.setBorder(BorderFactory.createMatteBorder(1, 0, 0, 0, Color.BLACK));
            Iterator it = myNode.getAttributes().listIterator();
            while(it.hasNext()){
            	_attributes.add(new JLabel(" @ " + ((EntityAttribute)it.next()).getName()));
            }
            if(_attributes.getComponentCount() == 0 || !Easik.getInstance().getFrame().getShowAttsVal())
            	_attributes.setVisible(false);
            
            //Add in unique keys
            this.add(_uniqueKeys = new JPanel(new GridLayout(0,1)));
            _uniqueKeys.setBorder(BorderFactory.createMatteBorder(1, 0, 0, 0, Color.BLACK));
            Iterator it2 = myNode.getUniqueKeys().listIterator();
            while(it2.hasNext()){
            	_uniqueKeys.add(new JLabel(" $ " + ((UniqueKey)it2.next()).getKeyName()));
            }
            if(_uniqueKeys.getComponentCount() == 0 || !Easik.getInstance().getFrame().getShowAttsVal())
            	_uniqueKeys.setVisible(false);
            
            //Set desired size
            this.setSize(this.getPreferredSize());
            
            return this;
        }
		
        /**
         * Takes the graph constants assigned and uses them with this class
         * 
         * @param graph The graph in use
         * @param attributes The attributes in use
         */
		protected void installAttributes(JGraph graph, Map attributes) {
            setOpaque(GraphConstants.isOpaque(attributes));
            Color foreground = GraphConstants.getForeground(attributes);
            setForeground((foreground != null) ? foreground : graph.getForeground());
            Color background = GraphConstants.getBackground(attributes);
            setBackground((background != null) ? background : graph.getBackground());
            Font font = GraphConstants.getFont(attributes);
            setFont((font != null) ? font : graph.getFont());
            Border border= GraphConstants.getBorder(attributes);
            Color bordercolor = GraphConstants.getBorderColor(attributes);
            if(border != null){
                setBorder(border);
            }
            else if (bordercolor != null) {
                int borderWidth = Math.max(1, Math.round(GraphConstants.getLineWidth(attributes)));
                setBorder(BorderFactory.createLineBorder(bordercolor, borderWidth));
            }
            else if (bordercolor == null)//case with no border
            {
            	setBorder(BorderFactory.createLineBorder(null, 0));
            }
        }
		
		/**
		 * Draws the object on the screen
		 * 
		 * @param g The graphics to be drawn
		 */
		public void paint(Graphics g) {
    		try {
    			if (gradientColor != null && !preview) {
    				setOpaque(false);
    				Graphics2D g2d = (Graphics2D) g;
    				g2d.setPaint(new GradientPaint(0, 0, getBackground(), getWidth(),
    						getHeight(), gradientColor, true));
    				g2d.fillRect(0, 0, getWidth(), getHeight());
    			}
    			super.paint(g);
    			paintSelectionBorder(g);
    		} catch (IllegalArgumentException e) {
    			// JDK Bug: Zero length string passed to TextLayout constructor
    		}
    	}

    	/**
    	 * Provided for subclassers to paint a selection border.
    	 * 
    	 * @param g Graphic to be drawn
    	 */
    	protected void paintSelectionBorder(Graphics g) {
    		((Graphics2D) g).setStroke(GraphConstants.SELECTION_STROKE);
    		if (hasFocus && selected)
    			g.setColor(graph.getLockedHandleColor());
    		else if (selected)
    			g.setColor(graph.getHighlightColor());
    		if (selected) {
    			Dimension d = getSize();
    			g.drawRect(0, 0, d.width - 1, d.height - 1);
    		}
    	}
    	
    	/**
    	 * Returns the intersection of the bounding rectangle and the straight line
    	 * between the source and the specified point p. The specified point is
    	 * expected not to intersect the bounds.
    	 * 
    	 * @param view The view of the cell
    	 * @param source The source of the line
    	 * @param p The inner point which the line should be drawn towards
    	 * @return The perimeter intersection point
    	 */
    	public Point2D getPerimeterPoint(VertexView view, Point2D source, Point2D p) {
    		Rectangle2D bounds = view.getBounds();
    		double x = bounds.getX();
    		double y = bounds.getY();
    		double width = bounds.getWidth();
    		double height = bounds.getHeight();
    		double xCenter = x + width / 2;
    		double yCenter = y + height / 2;
    		double dx = p.getX() - xCenter; // Compute Angle
    		double dy = p.getY() - yCenter;
    		double alpha = Math.atan2(dy, dx);
    		double xout = 0, yout = 0;
    		double pi = Math.PI;
    		double pi2 = Math.PI / 2.0;
    		double beta = pi2 - alpha;
    		double t = Math.atan2(height, width);
    		if (alpha < -pi + t || alpha > pi - t) { // Left edge
    			xout = x;
    			yout = yCenter - width * Math.tan(alpha) / 2;
    		} else if (alpha < -t) { // Top Edge
    			yout = y;
    			xout = xCenter - height * Math.tan(beta) / 2;
    		} else if (alpha < t) { // Right Edge
    			xout = x + width;
    			yout = yCenter + width * Math.tan(alpha) / 2;
    		} else { // Bottom Edge
    			yout = y + height;
    			xout = xCenter + height * Math.tan(beta) / 2;
    		}
    		return new Point2D.Double(xout, yout);
    	}
	}
}

